/*  Schreibt die offtimes-Einträge einer Ressource um, wenn sich ein Tag von einem Arbeitsfreiem Tag zu einem Arbeitstag oder
    umgekehrt wandelt.

    Dies passiert, wenn ein arbeitsfreier Tag (z.B. am Wochende) eine Kapazität in der Plantafel zugewiesen bekommt oder diese
    wieder gelöscht wird.
*/
SELECT tsystem.function__drop_by_regex( 'resource_timeline__rewrite_offtime', 'scheduling', _commit => true );
CREATE OR REPLACE FUNCTION scheduling.resource_timeline__rewrite_offtime(
      _resource_id  int,
      _date         date,
      _toworkday    bool  DEFAULT true,
      _loglevel    int   DEFAULT TSystem.Log_Get_LogLevel( _user => 'yes' )
  ) RETURNS void AS $$
  DECLARE

      _resource_record scheduling.resource;
      _shift_times  time[2];
      _day_before_is_off_day bool := false;
      _day_after_is_off_day bool := false;

      _prefix varchar := format( 'scheduling.resource_timeline__rewrite_offtime resource_id:%L, date:%L -', _resource_id, _date );

  BEGIN
      -- Debug
      IF _loglevel >= 4 THEN
          RAISE NOTICE '%', format(
                                      'call: scheduling.resource_timeline__rewrite_offtime( _resource_id => %L, _date => %L, _toworkday => %L, _loglevel => %L )',
                                                                                            _resource_id      , _date      , _toworkday      , _loglevel
                            )
          ;
      END IF;

      -- Daten der übergebenen Resource auslesen.
      _resource_record := r FROM scheduling.resource AS r WHERE id = _resource_id;

      -- Übergebene ressource prüfen.
      IF ( _resource_record IS NULL ) THEN
          RAISE EXCEPTION '%', format( 'resource id %L not found', _resource_id );
      END IF;
      IF ( _resource_record.context <> 'ksvba' ) THEN
          RAISE EXCEPTION 'rewrite offtime only for ksvba implemented';
      END IF;

      -- Prüfen ob, der Vortag ein arbeitsfreier Tag ist.
      IF EXISTS(
                    SELECT ti_id
                      FROM scheduling.resource_timeline
                     WHERE
                              ti_resource_id = _resource_id
                          AND ti_type = 'off.day'::scheduling.resource_timeline_blocktype
                          AND ti_date_end = _date::timestamp
               )
      THEN
          _day_before_is_off_day := true;
      END IF;

      -- Prüfen ob, der nächste Tag ein arbeitsfreier Tag ist.
      IF EXISTS(
                    SELECT ti_id
                      FROM scheduling.resource_timeline
                     WHERE
                              ti_resource_id = _resource_id
                          AND ti_type = 'off.day'::scheduling.resource_timeline_blocktype
                          AND ti_date_start = _date::timestamp  + interval '1 day'
               )
      THEN
          _day_after_is_off_day := true;
      END IF;

      -- Ermittle die Schichtzeiten des zur Ressource gehörenden Arbeitsplatzes.
      _shift_times := scheduling.ksvba__get_shifttimes( _resource_record.context_id );

      -- Debug
      IF _loglevel >= 5 THEN
          RAISE NOTICE '% _day_before_is_off_day:%, _day_after_is_off_day:%, _shift_times:%;', _prefix, _day_before_is_off_day, _day_after_is_off_day, _shift_times;
      END IF;


      -- Der übergebene Tag wird auf der genannten Ressource zu einem Arbeitstag bzw bleibt ein Arbeitstag.
      IF _toworkday = true THEN

          -- Lösche den eventuell vorhandenen off.day-Eintrag an dem übergebenen Tag.
          DELETE FROM scheduling.resource_timeline
          WHERE
                    ti_resource_id = _resource_id
                AND ti_type = 'off.day'::scheduling.resource_timeline_blocktype
                AND ti_date_start = _date::timestamp
                AND ti_date_end   = _date::timestamp + interval '1 day';

          -- Beginn der Arbeitszeit. Deshalb Betrachtung des Vortages.
          -- Vortag ist eine arbeitsfreier Tag
          IF _day_before_is_off_day IS true THEN
              -- Lösche Off.Time vom Vortag zum übergebenen Tag.
              DELETE FROM scheduling.resource_timeline
              WHERE
                        ti_resource_id = _resource_id
                    AND ti_type = 'off.time'::scheduling.resource_timeline_blocktype
                    AND ti_date_end   >= _date::timestamp                         -- Ende der Ausschaltzeit endet 0 Uhr oder später am gegebenen Tag.
                    AND ti_date_end   <  _date::timestamp + interval '1 day';
              -- Schichtbeginn ist 0:00 Uhr.
              IF _shift_times[1] = '0:00:00'::time THEN
                  -- Wegen dem Schichtbeginn von 0:00 Uhr hätte die Ausschaltzeit die Länge 0. Es ist nichts zu tun.
                  NULL;
              -- Schichtbeginn ist NACH 0:00 Uhr.
              ELSE
                  -- Erstelle Ausschaltzeit von 0.00 Uhr bis Begin der Arbeitszeit ein.
                  INSERT INTO scheduling.resource_timeline( ti_resource_id, ti_date_start   , ti_date_end                           , ti_type    )
                  VALUES                                  ( _resource_id  , _date::timestamp, ( _date + _shift_times[1] )::timestamp, 'off.time' );
              END IF;
          -- Vortag ist ein Arbeitstag
          ELSE
              -- Lösche Off.Time vom Vortag zum übergebenen Tag.
              DELETE FROM scheduling.resource_timeline
              WHERE
                        ti_resource_id = _resource_id
                    AND ti_type = 'off.time'::scheduling.resource_timeline_blocktype
                    AND ti_date_end   >= _date::timestamp                         -- Ende der Ausschaltzeit endet 0 Uhr oder später am gegebenen Tag.
                    AND ti_date_end   <  _date::timestamp + interval '1 day';
              -- Erstelle Ausschaltzeit vom Vortag bis zum Begin der Arbeitszeit.
              INSERT INTO scheduling.resource_timeline( ti_resource_id, ti_date_start                                            , ti_date_end                           , ti_type    )
              VALUES                                  ( _resource_id  , ( _date + _shift_times[2] )::timestamp - interval '1 day', ( _date + _shift_times[1] )::timestamp, 'off.time' );
          END IF;

          -- Ende der Arbeitszeit. Deshalb Betrachtung des Folgetages.
          -- Lösche Off.Time zum Folgetag.
          DELETE FROM scheduling.resource_timeline
          WHERE
                    ti_resource_id = _resource_id
                AND ti_type = 'off.time'::scheduling.resource_timeline_blocktype
                AND ti_date_start >  _date::timestamp                         -- Beginn der Ausschaltzeit beginnt nach 0 Uhr oder später am gegebenen Tag.
                AND ti_date_start <= _date::timestamp + interval '1 day';
          -- Folgetag ist ein arbeitsfreier Tag
          IF _day_after_is_off_day IS true THEN
              -- Erstelle Ausschaltzeit vom Ende der Arbeitszeit bis zum Folgetag 0:00 Uhr.
              INSERT INTO scheduling.resource_timeline( ti_resource_id, ti_date_start                         , ti_date_end                        , ti_type    )
              VALUES                                  ( _resource_id  , ( _date + _shift_times[2] )::timestamp, _date::timestamp + interval '1 day', 'off.time' );
          -- Folgetag ist ein Arbeitstag
          ELSE
              -- Erstelle Ausschaltzeit vom Ende der Arbeitszeit bis zum Folgetag.
              INSERT INTO scheduling.resource_timeline( ti_resource_id, ti_date_start                         , ti_date_end                                              , ti_type    )
              VALUES                                  ( _resource_id  , ( _date + _shift_times[2] )::timestamp, ( _date + _shift_times[1] )::timestamp + interval '1 day', 'off.time' );
          END IF;

      -- Der übergebene Tag wird auf der genannten Ressource zu einem arbeitsfreiem Tag oder bleibt ein  arbeitsfreier Tag.
      ELSE
          -- Beginn der Arbeitszeit. Deshalb Betrachtung des Vortages.
          -- Off.Time-Eintrag am Beginn der Arbeistzeit löschen.
          DELETE FROM scheduling.resource_timeline
          WHERE
                    ti_resource_id = _resource_id
                AND ti_type = 'off.time'::scheduling.resource_timeline_blocktype
                AND ti_date_end   >= _date::timestamp
                AND ti_date_end   <  _date::timestamp + interval '1 day';
          -- Vortag ist ein arbeitsfreier Tag.
          IF _day_before_is_off_day IS true THEN
              -- Von arbeitsfreiem Vortag zum arbeitsfreiem aktuellen Tag wird keine off.time benötigt. Es ist nichts zu tun.
              NULL;
          -- Vortag ist ein Arbeitstag
          ELSE
              -- Erstelle Ausschaltzeit vom Vortag bis 0:00 Uhr.
              INSERT INTO scheduling.resource_timeline( ti_resource_id, ti_date_start                                            , ti_date_end     , ti_type    )
              VALUES                                  ( _resource_id  , ( _date + _shift_times[2] )::timestamp - interval '1 day', _date::timestamp, 'off.time' );
          END IF;

          -- Ende der Arbeitszeit. Deshalb Betrachtung des Folgetages.
          -- Off.Time-Eintrag am Ende der Arbeitszeit löschen.
          DELETE FROM scheduling.resource_timeline
          WHERE
                    ti_resource_id = _resource_id
                AND ti_type = 'off.time'::scheduling.resource_timeline_blocktype
                AND ti_date_start >  _date::timestamp
                AND ti_date_start <= _date::timestamp + interval '1 day';
          -- Folgetag ist ein arbeitsfreier Tag oder der Schichtbeginn ist 0:00 Uhr.
          IF _day_after_is_off_day IS true OR _shift_times[1] = '0:00:00'::time THEN
              -- Von arbeitsfreiem aktuellen Tag zum arbeitsfreiem Folgetag wird keine off.time benötigt bzw. wegen dem Schichtbeginn von 0:00 Uhr hätte die Ausschaltzeit die Länge 0. Es ist nichts zu tun.
              NULL;
          -- Folgetag ist ein Arbeitstag
          ELSE
              -- Erstelle Ausschaltzeit von 24:00 Uhr bis zum Folgetag.
              INSERT INTO scheduling.resource_timeline( ti_resource_id, ti_date_start                      , ti_date_end                                              , ti_type    )
              VALUES                                  ( _resource_id  , _date::timestamp + interval '1 day', ( _date + _shift_times[1] )::timestamp + interval '1 day', 'off.time' );
          END IF;

          -- Trage einen off.day-Eintrag an dem übergebenen Tag ein.
          IF NOT EXISTS( SELECT true FROM scheduling.resource_timeline WHERE ti_resource_id = _resource_id AND ti_type = 'off.day' AND ti_date_start = _date::timestamp ) THEN
              INSERT INTO scheduling.resource_timeline( ti_resource_id, ti_date_start   , ti_date_end                        , ti_type   )
              VALUES                                  ( _resource_id  , _date::timestamp, _date::timestamp + interval '1 day', 'off.day' );
          END IF;

      END IF;

  END $$ language plpgsql;
